//////////////////////////////////////////////////////////////////  
// cran32.cpp: Class implementation for a combined 32 bit
// linear congruential uniform random number generator.
// Copyright (c) 1994 Azarona Software. All rights reserved. 
// NOTE: This generator only works when long is 32 bits or more.  
////////////////////////////////////////////////////////////////// 

#include "cran32.h" 
// #include <iostream.h>
 
// NOTE: These constants are designed to work together. 
// Don't change them unless you know what you are doing. 
 
// Constants for first generator 

const long CRan32::m1 = 2147483563L; 
const long CRan32::a1 = 40014L; 
const long CRan32::q1 = m1 / a1; 
const long CRan32::r1 = m1 % a1; 
 
// Constants for second generator 

const long CRan32::m2 = 2147483399L; 
const long CRan32::a2 = 40692L; 
const long CRan32::q2 = m2 / a2; 
const long CRan32::r2 = m2 % a2; 
 
void CRan32::Reset(long s1, long s2) 
// Resets the random number generator by setting 
// the two seeds. The seeds must be between  
// [1 .. respective_modulo - 1]. 
{ 
  seed1 = s1; 
  if (seed1 < 1) seed1 = 1; 
  else if (seed1 >= m1) seed1 = m1-1; 
 
  seed2 = s2; 
  if (seed2 < 1) seed2 = 1; 
  else if (seed2 >= m2) seed2 = m2-1; 
 
  // Combine the seeds to get the current value 
 
  curr = seed1 - seed2; 
  if (curr < 1) curr += m1-1; 
} 
 
long CRan32::Next() 
// Advances to the next random number with uniform distribution, 
// with a range of [1..m1-1]. The period is (m1-1)*(m2-1)/2,  
// or roughly 2.3x10^18 (huge!) for the constants as given.  
{ 
  long k; 
 
  // Run the two generators independently 
 
  k = seed1 / q1; 
  seed1 = a1 * (seed1 - k*q1) - k*r1; 
  if (seed1 < 0) seed1 += m1; 
 
  k = seed2 / q2; 
  seed2 = a2 * (seed2 - k*q2) - k*r2; 
  if (seed2 < 0) seed2 += m2; 
 
  // Combine their results 
 
  curr = seed1 - seed2; 
  if (curr < 1) curr += m1-1; 
   
  //cout << curr << "\t";
  return curr; 
} 
 
long CRan32::Curr(long lo, long hi) const 
// Returns the current value of the sequence, cv, and 
// restricts it's range to be lo <= cv <= hi. 
{ 
  return lo + (Curr() % (hi-lo+1)); 
} 
 
long CRan32::Next(long lo, long hi) 
// Returns the next value of the sequence, nv, and 
// restricts it's range to be lo <= nv <= hi. 
{ 
  return lo + (Next() % (hi-lo+1)); 
} 
